/*
	WinSocket.cpp
	This file serves as the wrapper for all sockets in *NIX.
	Copyright 2002 Ben Everett and Chris Horlick.
	All Rights Reserved.
*/

#include "pch.h"


#ifdef _WIN32
// Windows

// Engine Includes
#include "Socket.h"
#include "WinSocket.h"

using namespace CPPMUD;

// Constructor & Deconstructor
CWinSocket::CWinSocket()
{
}

CWinSocket::~CWinSocket()
{
}

// Functions
ErrRet CWinSocket::CloseClient(int nClientID)
/*
	Closes a client's connection by their client descriptor.
	PRE: Server has been initialized properly, pClient is a valid pointer to a client descriptor.
	POST: Client's connection closed.
*/
{
	std::vector<CClientDescriptor>::iterator itRem = m_vecClients.begin();

	// Remove the client from the client list.
	for (int i = 0; i < m_vecClients.size(); ++i, ++itRem)
	{
		if (nClientID == m_vecClients[i].m_nClientID)
		{
			std::cout << "Closing client #" << m_vecClients[i].m_nClientID << "'s connection." << 
				std::endl;

			// Close the socket.
			shutdown(m_vecClients[i].m_sSocket, SD_BOTH);
			closesocket(m_vecClients[i].m_sSocket);

			m_vecClients.erase(itRem);
		}
	}

	return errNone;
}

ErrRet CWinSocket::CloseSocket(SOCKET sSocket)
/*
	Closes a client's connection by their socket descriptor.
	PRE: Server has been initialized properly, sSocket is a valid socket descriptor.
	POST: Client's connection closed.
*/
{
	std::cout << "Closing socket, connection timed out." << std::endl;

	// Close the socket.
	shutdown(sSocket, SD_BOTH);
	closesocket(sSocket);

	return errNone;
}

ErrRet CWinSocket::CloseSockets()
/*
	Closes all client sockets and finally the server socket.
	PRE: Server was initialized properly.
	POST: All sockets closed properly.
*/
{
	// Close all client's connections.
	for (int i = 0; i < m_vecClients.size(); ++i)
	{
		shutdown(m_vecClients[i].m_sSocket, SD_BOTH);
		closesocket(m_vecClients[i].m_sSocket);
	}

	// Close the server socket so no-one else can connect.
	shutdown(m_sSocket, SD_BOTH);
	closesocket(m_sSocket);

	// Clear the client list.
	m_vecClients.clear();

	return errNone;
}

ErrRet CWinSocket::Connect(int nPort)
/*
	Connects to a server on the given port.
	PRE: Server socket has been initialized properly, nPort is a valid port.
	POST: Server is connected to a remote server.
*/
{
	return errNone;
}

ErrRet CWinSocket::InitSockets()
/*
	Initializes the server socket.
	PRE: None.
	POST: Server socket initialized.
*/
{
	// Start WinSock.
	if (WSAStartup(0x0202, &m_wsaData))
	{
		printf("Failed.\nFailed to initialize WinSock 2.\n");

		return errSocketFailCreate;
	}

	// Make sure we are running WinSock v2.
	if (m_wsaData.wVersion != 0x0202)
	{
		printf("Failed.\nInvalid version of WinSock.\n");

		return errSocketFailCreate;
	}

	// Initialize the socket.
	m_sSocket = socket(AF_INET, SOCK_STREAM, 0);

	// Setup non-blocking on the socket.
	NonBlocking(m_sSocket);

	return errNone;
}

ErrRet CWinSocket::Listen(int nPort)
/*
	Listens for client connections on the given port.
	PRE: Server has been initialized properly, nPort is a valid port.
	POST: Server is listening for incoming client connections.
*/
{
	m_saAddress.sin_family = AF_INET;
	m_saAddress.sin_port = htons(nPort);
	m_saAddress.sin_addr.s_addr = htonl(INADDR_ANY);

	// Bind the socket.
	if (bind(m_sSocket, (LPSOCKADDR)&m_saAddress, sizeof(m_saAddress)) == SOCKET_ERROR)
	{
		printf("Failed.\nUnable to bind port.\n");

		return errSocketFailCreate;
	}

	// Listen for any incoming connections.
	if (listen(m_sSocket, 10) == SOCKET_ERROR)
	{
		printf("Failed.\nUnable to listen for incoming connections.\n");

		return errSocketFailListen;
	}

	return errNone;
}

ErrRet CWinSocket::ReadData(char *czData, int nBufferSize, int nClientID)
/*
	Reads data from the client descriptor and copies it into czData.
	PRE: Server has been initialized properly, czData is a valid pointer to the read buffer, 
		pClient is a valid pointer to a client descriptor, nBufferSize is the length of czData..
	POST: Client data has been read.
*/
{
	int nIndex = GetIndexFromClientID(nClientID);
	int nRead = 0;

	if (nIndex < 0)
		return errUnknownClient;
	
	nRead = recv(m_vecClients[nIndex].m_sSocket, czData, nBufferSize, 0);

	if (nRead < 0)
		return errSocketTimeOut;
	else if (nRead == 0)
		return errSocketFailRead;

	return errNone;
}

ErrRet CWinSocket::SendData(int nClientID)
/*
	Sends as much data as possible in nClientID's outgoing buffer.
	PRE: Server has been initialized properly, nClientID is a valid client ID.
	POST: As much data as possible is sent from the client's outgoing buffer.
*/
{
	int nIndex = GetIndexFromClientID(nClientID);
	int nSent = 0;
	int nLen = strlen(m_vecClients[nIndex].m_czOutgoingData);

	// Send as much data as possible to the client.
	nSent = send(m_vecClients[nIndex].m_sSocket, m_vecClients[nIndex].m_czOutgoingData, 
		strlen(m_vecClients[nIndex].m_czOutgoingData), 0);

	// If nSent is < 0 the socket timed out, if it = 0 there was an error somewhere, if > 0 nSent 
	//    removes that many char's from the outgoing buffer.
	if (nSent < 0)
		return errSocketTimeOut;
	else if (nSent == 0)
		return errSocketFailWrite;
	else
	{
		memmove(m_vecClients[nIndex].m_czOutgoingData, m_vecClients[nIndex].m_czOutgoingData + 
			nSent, 2048 - nSent);
		memset(m_vecClients[nIndex].m_czOutgoingData + (nLen - nSent), 0, sizeof(char) * (nLen - 
			nSent));
	}

	return errNone;
}

void CWinSocket::NonBlocking(SOCKET sSocket)
/*
	Sets the given socket to non-blocking mode.
	PRE: Server has been initialized properly, sSocket is a valid socket descriptor.
	POST: Socket set to non-blocking mode.
*/
{
	unsigned long lNonBlocking = 1;

	// Set the socket to non-blocking.
	ioctlsocket(sSocket, FIONBIO, &lNonBlocking);
}

#endif